#!/usr/bin/python

import wx

import socket, time, types
import struct, os

from Common import *
from Config import *
from Command import *

NDRP_DISCOVERY,	\
NDRP_SETENV,	\
NDRP_SETFILE,	\
NDRP_EXEC,	\
NDRP_CHDIR,	\
NDRP_MAX = range(1, 7)

NDRP_TYPE_PROBE,	\
NDRP_TYPE_INITIATE,	\
NDRP_TYPE_MAX = range(3)

ENV_TYPE_LIB,	\
ENV_TYPE_PATH,	\
ENV_TYPE_MAX = range(3)

DRV_TYPE_SINGLE,	\
DRV_TYPE_DUAL,	\
DRV_TYPE_MAX = range(3)

TRANS_MODE_RDONLY,	\
TRANS_MODE_WRONLY,	\
TRANS_MODE_RW,	\
TRANS_MODE_MAX = range(4)

NDRP_HDR_SIZ = 12

class NDRPException(Exception):
	pass

class NdrpPacketFactory(object):
	def __init__(self, logger, shower):
		self.logger = logger
		self.shower = shower
		self.classes = {
			1: NdrpPacketDiscovery,
			2: NdrpPacketSetenv,
			3: NdrpPacketSetFile,
			4: NdrpPacketExec,
			5: NdrpPacketChdir,
		}

	def parse(self, buffer):
		(seq, ts, opcode, datalen) = struct .unpack("!IfHH", buffer[0:12])
		wx.CallAfter(self.logger, 'Opcode: %d, Seq: %d, Datalen: %d\n' % (opcode, seq, datalen))
		try:
			packet = self.__create(opcode)
		except:
			# Bogus - Wrong Opcode
			raise
		packet.buffer = buffer
		return packet.decode()

	def __create(self, opcode):
		packet = self.classes[opcode](self.logger, self.shower)
		return packet

##############################################################################
import threading
class NdrpClient(threading.Thread):
	def __init__(self, logger, cb_exit, cb_show, type, slave, nasImage,
			port=NDRP_PROBE_PORT, timeout=NDRP_PROBE_TIMEO):
		threading.Thread.__init__(self)
		self.logger = logger
		self.type = type
		self.cb_exit = cb_exit
		self.cb_show = cb_show
		self.slave = slave
		self.nasImage = nasImage
		self.running = True

	def Log(self, msg):
		wx.CallAfter(self.logger, msg)

	def SetGauge(self, main, sub):
		wx.CallAfter(self.cb_show, main, sub)

	def kill(self):
		self.running = False

	def stop(self, retCode):
		wx.CallAfter(self.cb_exit, self.type, retCode)

	NumberOfSteps = 12
	def __Overall(self, step):
		return step * 100 / self.NumberOfSteps

	def run(self):
		if self.type == NDRP_TYPE_PROBE:
			self.Probe()
		elif self.type == NDRP_TYPE_INITIATE:
			(insType, ip, mac, ver, port) = self.slave
			self.Initiate(insType, ip, int(port), ver, self.nasImage)
		else:
			raise NDRPException, ('Unknown type', -ERR_INVALID)

	def __Transact(self, sock, type, sendbuf, mode=TRANS_MODE_RW, ignore=False):
		if self.running == False:
			raise NDRPException, ('User Abort', -ERR_ABORT)
		ndrp_factory = NdrpPacketFactory(self.logger, self.cb_show)
		if mode != TRANS_MODE_RDONLY:
			try:
				nbytes = sock.send(sendbuf)
			except socket.error, msg:
				self.Log('[%s] Send/Recv Error - %s\n' % (msg[0], msg[1]))
				sock.close()
				raise NDRPException, ('Socket Error', -ERR_SOCK)

		if mode == TRANS_MODE_WRONLY:
			return nbytes

		while True:
			recvbuf = sock.recv(MAX_BUFSIZ)
			self.Log('Received %d bytes response\n' % len(recvbuf))
			try:
				recvpkt = ndrp_factory.parse(recvbuf)
			except:
				# The remote closed the connection
				sock.close()
				raise NDRPException, ('Cannot decode the receiving packet', -ERR_DECODE)

			if isinstance(recvpkt, type):
				break

		if recvpkt.retCode != 0:
			self.Log('Runtime Execution Error: Code [%d]\n' % recvpkt.retCode)
			if ignore == False:
				sock.close()
			raise NDRPException, ('Runtime execution error\n', -ERR_EXEC)

		return recvpkt

	def Initiate(self, insType, ip, port, sVer, nasImage):

		if insType == DRV_TYPE_DUAL:
			drive = DUAL_DRIVE
		else:
			drive = SINGLE_DRIVE

		try:
			sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
			sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)
			sock.connect((ip, port))
		except socket.error, msg:
			self.Log('[%s] Failed to connect %s:%s - %s\n'
				% (msg[0], ip, port, msg[1]))
			sock.close()
			self.stop(-ERR_CONNECT)
			return
		self.Log('Connect to %s:%s\n' % (ip, port))
		# Start working hard here
		# Step 0: Cleanup
		pkt = NdrpPacketExec(self.logger, self.cb_show)
		for cmd in (CMD_CLEAN_ZOMBIES + CMD_UNMOUNT_ALL + CMD_STOP_RAID + CMD_CLEAN_WORKAREA):
			try:
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer, TRANS_MODE_RW, True)
			except:
				pass
		# Change dir to /tmp
		pkt = NdrpPacketChdir(self.logger, self.cb_show)
		try:
			recvpkt = self.__Transact(sock, NdrpPacketChdir,
				pkt.encode('/tmp').buffer)
			# Step 1: Set the running environment
			self.SetGauge((self.__Overall(0), 'Setting running environment'), (0, ''))
			pkt = NdrpPacketSetenv(self.logger, self.cb_show)
			for env_type in range(ENV_TYPE_MAX):
				recvpkt = self.__Transact(sock, NdrpPacketSetenv,
					pkt.encode(env_type).buffer)
				self.SetGauge(None, ((env_type+1)*100/ENV_TYPE_MAX, ''))
			# Step 2: Create Partitions
			self.SetGauge((self.__Overall(1), 'Creating Partitions'), (0, ''))
			pkt = NdrpPacketExec(self.logger, self.cb_show)
			for d in drive:
				progress = 1
				for cmd in CMD_CREAT_PART:
					if cmd != 'wait':
						cmd = cmd % d
					recvpkt = self.__Transact(sock, NdrpPacketExec,
						pkt.encode(cmd).buffer)
					self.SetGauge(None, (progress*100/len(CMD_CREAT_PART),
						'Partition %s' % d))
					progress += 1
			# Sleep 2 secs
			cmd = CMD_SLEEP % 2
			recvpkt = self.__Transact(sock, NdrpPacketExec,
				pkt.encode(cmd).buffer)
			# Step 3: Write boot rom directions
			self.SetGauge((self.__Overall(2), 'Writing BootROM Directions'), (0, ''))
			pkt = NdrpPacketSetFile(self.logger, self.cb_show)
			rom_code_size = len(BOOTROM_CODES) + BOOTROM_OFFSET
			send_bytes = self.__Transact(sock, NdrpPacketSetFile,
				pkt.encode(BOOTROM_TMP_NAME, rom_code_size).buffer)
			# Generate ROM codes and send out
			rom_codes = EncodeBinary(BOOTROM_OFFSET, 0x00, BOOTROM_CODES)
			send_bytes = self.__Transact(sock, NdrpPacketSetFile, rom_codes,
				TRANS_MODE_WRONLY)
			if send_bytes != rom_code_size:
				self.Log('Transfer file error! Only %d bytes transferred\n' % send_bytes)
				self.stop(-ERR_TRANS)
				return
			# Read the return value
			recvpkt = self.__Transact(sock, NdrpPacketSetFile,
				None, TRANS_MODE_RDONLY)
			# Write Codes
			pkt = NdrpPacketExec(self.logger, self.cb_show)
			progress = 1
			for d in drive:
				cmd = CMD_WRITE_BM_CODES % (BOOTROM_TMP_NAME, d)
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer)
				self.SetGauge(None, (progress*100/len(drive), ''))
				progress += 1
			# Step 4: Start Raid
			self.SetGauge((self.__Overall(3), 'Starting Raid'), (0, ''))
			drive_id = 0
			progress = 1
			for cmd in CMD_START_RAID:
				drive_id += 1
				if insType == DRV_TYPE_DUAL:
					cmd = cmd % (DUAL_DRIVE[0] + str(drive_id), DUAL_DRIVE[1] + str(drive_id))
				else:
					cmd = cmd % (DUAL_DRIVE[0] + str(drive_id), 'missing')
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer)
				self.SetGauge(None, (progress*100/(len(cmd)+1), ''))
				progress += 1
			# Step 4-1: Start Raid -- Data Partition
			if insType == DRV_TYPE_DUAL:
				cmd = CMD_START_PART4_RAID_DUAL % (DUAL_DRIVE[0]+'4',DUAL_DRIVE[1]+'4')
			else:
				cmd = CMD_START_PART4_RAID_SINGLE % (DUAL_DRIVE[0]+'4', 'missing')
			recvpkt = self.__Transact(sock, NdrpPacketExec,
				pkt.encode(cmd).buffer)
			self.SetGauge(None, (100, ''))
			# Step 5: Format Data Partition and Mount
			# Comment: At this moment, we don't have mkfs.ext3 yet, so format 
			# the data partition only. After formatting the partition, image
			# file can be uploaded
			self.SetGauge((self.__Overall(4), 'Create Temp Partition'), (0, ''))
			CMD_SEQ = [CMD_FORMAT_PART4, CMD_MOUNT_PART4]
			progress = 1
			for cmd in CMD_SEQ:
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer)
				self.SetGauge(None, (progress*100/len(CMD_SEQ), ''))
				progress += 1
			# Step 6-0: Change dir to /mnt/disk
			pkt = NdrpPacketChdir(self.logger, self.cb_show)
			recvpkt = self.__Transact(sock, NdrpPacketChdir,
				pkt.encode(MOUNT_DISK_BASE).buffer)
			# Step 6: Upload File
			self.SetGauge((self.__Overall(5), 'Uploading Image'), (0, ''))
			pkt = NdrpPacketSetFile(self.logger, self.cb_show)
			nasImageInfo = os.stat(self.nasImage)
			fname = str(os.path.basename(self.nasImage))
			send_bytes = self.__Transact(sock, NdrpPacketSetFile,
				pkt.encode(fname, nasImageInfo.st_size).buffer)
			# Transfer File
			try:
				fp = open(self.nasImage, "rb")
				try:
					imgSize = nasImageInfo.st_size
					send_bytes = 0
					while True:
						rdbuf = fp.read(MAX_BUFSIZ)
						if rdbuf == '':
							break
						send_bytes += self.__Transact(sock, NdrpPacketSetFile,
							rdbuf, TRANS_MODE_WRONLY)
						progress = send_bytes * 100 / imgSize
						self.SetGauge(None, (progress, ''))
				except IOError, (code, msg):
					self.Log('[Error %d] %s\n' % (code, msg))
				except NDRPException, (msg, errno): 
					fp.close()
					self.stop(errno)
					return
				finally:
					fp.close()
					if send_bytes != imgSize:
						self.Log('Transfer file error! Only %d bytes transferred\n' % send_bytes)
						self.stop(-ERR_TRANS)
						return
			except IOError, (code, msg):
				self.Log('[Error %d] %s\n' % (code, msg))
				self.stop(-ERR_FILE)
				return
			recvpkt = self.__Transact(sock, NdrpPacketSetFile,
				None, TRANS_MODE_RDONLY)
			# Step 7: Decompressing image
			self.SetGauge((self.__Overall(6), 'Decompressing Image'), (0, ''))
			pkt = NdrpPacketExec(self.logger, self.cb_show)
			CMD_SEQ = [CMD_UNTAR_IMAGE % fname, CMD_UNZIP_ROOTFS]
			time_lapsed = 1
			for cmd in CMD_SEQ:
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer, TRANS_MODE_WRONLY)
				# Actually, we don't know the progress of decompressing. So, we are cheating customers. According to the experiment, the decompressing performance is about 266480 Bps on 810. Here we set the timeout value to 1 sec. When the timer expires which means 810 has processed 266480 bytes already. So, the calculation of progress would be: 'time_lapsed * 266480 / original_file_size'
				sock.settimeout(1)
				while True:
					try:
						recvpkt = self.__Transact(sock, NdrpPacketExec,
							pkt.encode(cmd).buffer, TRANS_MODE_RDONLY)
					except socket.timeout, err:
						# Although the image file contains uImage, installation scripts, u-boot, and rootfs, the size of rootfs is far larger than the others. So, Use imgSize * 2 instead
						progress = DECOMP_ABILITY*100*time_lapsed/(imgSize*2)
						if progress > 99: progress = 99
						self.SetGauge(None, (progress,''))
						time_lapsed += 1
						continue
					break
				sock.settimeout(None)
			self.SetGauge(None, (100,''))
			# Step 8: Mount Root File System
			self.SetGauge((self.__Overall(7), 'Mounting Root File System'), (0, ''))
			progress = 1
			for cmd in CMD_MOUNT_ROOTFS:
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer)
				self.SetGauge(None, (progress*100/len(CMD_MOUNT_ROOTFS), ''))
				progress += 1
			# Step 9: Format System Partitions
			self.SetGauge((self.__Overall(8), 'Formatting Partitions'), (0, ''))
			progress = 1
			for cmd in CMD_FORMAT_PART:
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer)
				self.SetGauge(None, (progress*100/len(CMD_FORMAT_PART), ''))
				progress += 1
			# Step 10: Copy Root File System 
			self.SetGauge((self.__Overall(9), 'Creating Root File System'), (0, ''))
			progress = 1
			if insType == DRV_TYPE_DUAL:
				cmdSysType = [CMD_SET_DUAL, CMD_SYS_TYPE % SYS_2NC]
			else:
				cmdSysType = [CMD_SYS_TYPE % SYS_1NC,]
			sysType = CMD_SYS_TYPE
			CMD_SEQ = CMD_COPY_ROOTFS +	\
				cmdSysType + CMD_COPY_ROOTFS_UNMOUNT
			for cmd in CMD_SEQ:
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer)
				self.SetGauge(None, (progress*100/len(CMD_SEQ), ''))
				progress += 1
			# Step 11: Write hidden sectors
			self.SetGauge((self.__Overall(10), 'Write Hidden Sectors'), (0, ''))
			for d in drive:
				progress = 1
				for cmd in CMD_WR_SECT:
					cmd = cmd % (IMAGE_PATH, d)
					recvpkt = self.__Transact(sock, NdrpPacketExec,
						pkt.encode(cmd).buffer)
					self.SetGauge(None, (progress*100/len(CMD_WR_SECT), ''))
					progress += 1
			# Step 12-0: Change dir to /tmp to make the device unmountable
			pkt = NdrpPacketChdir(self.logger, self.cb_show)
			recvpkt = self.__Transact(sock, NdrpPacketChdir,
				pkt.encode('/tmp').buffer)
			# Step 12: Finialization - Stop Raid
			self.SetGauge((self.__Overall(11), 'Finishing the script'), (0, ''))
			progress = 1
			pkt = NdrpPacketExec(self.logger, self.cb_show)
			CMD_SEQ = CMD_FINALIZE + CMD_STOP_RAID
			for cmd in CMD_SEQ:
				recvpkt = self.__Transact(sock, NdrpPacketExec,
					pkt.encode(cmd).buffer)
				self.SetGauge(None, (progress*100/len(CMD_SEQ), ''))
				progress += 1
		except NDRPException, (msg, errno):
			self.stop(errno)
			return
		self.SetGauge((self.__Overall(12), 'Finishing the script'), (0, ''))
		cmd = CMD_REBOOT
		recvpkt = self.__Transact(sock, NdrpPacketExec,
			pkt.encode(cmd).buffer)
		sock.close()
		self.stop(ERR_OK)

	
	def Probe(self, timeout=NDRP_PROBE_TIMEO):
		recvpkt = None
		start_time = time.time()
		bytes = 0

		ndrp_factory = NdrpPacketFactory(self.logger, self.cb_show)
		sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
		sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)

		self.Log('Sending NDRP Probe\n')

		pkt = NdrpPacketDiscovery(self.logger, self.cb_show)
		probe_addr = GetBcastAddr()
		for addr in probe_addr:
			self.Log('Probing Address: %s\n' % addr)
			try:
				sock.sendto(pkt.encode().buffer, (addr, NDRP_PROBE_PORT))
			except socket.error, msg:
				self.Log('[%s] Sendto Error - %s\n' % (msg[0], msg[1]))
		
		while True:
			if timeout <= 0:
				break
			sock.settimeout(timeout)
			start_time = time.time()
			try:
				(buffer, (raddress, rport)) = sock.recvfrom(MAX_BUFSIZ)
			except socket.timeout, err:
				self.Log('Probing Finished\n')
				break
			timeout -= (time.time() - start_time)

			self.Log('Received %d bytes from %s %s\n' %
				(len(buffer), raddress, rport))
			try:
				recvpkt = ndrp_factory.parse(buffer)
			except:
				pass

		sock.close()
		self.stop(ERR_OK)

import random, sys
class NdrpPacket(object):
	seq = random.randint(0, sys.maxint)
	def __init__(self, logger, shower, opcode):
		self.logger = logger
		self.shower = shower
		self.opcode = opcode
		self.buffer = None

	def encode(self, data):
		format = "!IfHH"
		if data is None:
			self.buffer = struct.pack(format, self.seq, time.time(),
				self.opcode, 0)
		else:
			format += "%dsx" % len(data)
			self.buffer = struct.pack(format, self.seq, time.time(),
				self.opcode, len(data) + 1, data)
		self.seq += 1
		return self

	def decode(self):
		raise NotImplementedError, "Abstract method"

class NdrpPacketDiscovery(NdrpPacket):
	def __init__(self, logger, shower):
		NdrpPacket.__init__(self, logger, shower, NDRP_DISCOVERY)

		self.logger = logger
		self.shower = shower

	def encode(self):
		return NdrpPacket.encode(self, None)

	def decode(self):
		format = "!IfHH"
		(self.seq, self.ts, self.opcode, self.datalen) = struct.unpack(format, self.buffer[0:12])
		if not self.datalen:
			self.message = ''
		else:
			format = "%ds" % self.datalen
			self.message = struct.unpack(format, self.buffer[12:])
		wx.CallAfter(self.logger, 'Receiving Probing Response : %s\n' % self.message)
		if self.message:
			(mac, ip, port, ver) = self.message[0].split(':')
			wx.CallAfter(self.shower, ip, mac, ver, port)
		return self

class NdrpPacketSetenv(NdrpPacket):
	ENV_TYPE = ['LD_LIBRARY_PATH', 'PATH']
	def __init__(self, logger, shower):
		NdrpPacket.__init__(self, logger, shower, NDRP_SETENV)

		self.logger = logger
		self.shower = shower

	def encode(self, env_type):
		envStr = self.ENV_TYPE[env_type] + '='
		if env_type == ENV_TYPE_LIB:
			target = ENV_LIB
		else:
			target = ENV_PATH
		delim = False
		for e in target:
			if delim:
				envStr += ':'
			envStr += e + ':' + MOUNT_IMAGE_BASE + e
			delim = True
		return NdrpPacket.encode(self, envStr)

	def decode(self):
		(self.retCode, ) = struct.unpack('!i', self.buffer[12:])
		return self

class NdrpPacketSetFile(NdrpPacket):
	def __init__(self, logger, shower):
		NdrpPacket.__init__(self, logger, shower, NDRP_SETFILE)

		self.logger = logger
		self.shower = shower

	def encode(self, fname, len):
		return NdrpPacket.encode(self, (fname + ':' + str(len)))

	def decode(self):
		(self.retCode, ) = struct.unpack('!i', self.buffer[12:])
		return self

class NdrpPacketExec(NdrpPacket):
	def __init__(self, logger, shower):
		NdrpPacket.__init__(self, logger, shower, NDRP_EXEC)

		self.logger = logger
		self.shower = shower

	def encode(self, cmd):
		return NdrpPacket.encode(self, cmd)

	def decode(self):
		(self.retCode, ) = struct.unpack('!i', self.buffer[12:])
		return self

class NdrpPacketChdir(NdrpPacket):
	def __init__(self, logger, shower):
		NdrpPacket.__init__(self, logger, shower, NDRP_CHDIR)

		self.logger = logger
		self.shower = shower

	def encode(self, dir):
		return NdrpPacket.encode(self, dir)

	def decode(self):
		(self.retCode, ) = struct.unpack('!i', self.buffer[12:])
		return self

